import Image from "next/image"
import Link from "next/link"
import { Card, CardContent } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import { Star, Calendar, Shield } from "lucide-react"

interface TVShow {
  id: number
  name: string
  poster_path: string | null
  first_air_date: string
  vote_average: number
  overview: string
  adult?: boolean
}

interface TVGridProps {
  search?: string
  genre?: string
  year?: string
  sort?: string
  page?: number
  showAdultContent?: boolean
}

async function getTVShows({
  search,
  genre,
  year,
  sort = "popularity.desc",
  page = 1,
  showAdultContent = false,
}: TVGridProps) {
  const baseUrl = "https://api.themoviedb.org/3"
  let url = ""

  if (search) {
    url = `${baseUrl}/search/tv?api_key=${process.env.TMDB_API_KEY}&query=${encodeURIComponent(search)}&page=${page}&include_adult=${showAdultContent}`
  } else {
    url = `${baseUrl}/discover/tv?api_key=${process.env.TMDB_API_KEY}&page=${page}&sort_by=${sort}&include_adult=${showAdultContent}`

    if (genre && genre !== "All Genres") {
      // Map genre name to ID
      const genreMap: { [key: string]: number } = {
        "Action & Adventure": 10759,
        Animation: 16,
        Comedy: 35,
        Crime: 80,
        Documentary: 99,
        Drama: 18,
        Family: 10751,
        Kids: 10762,
        Mystery: 9648,
        News: 10763,
        Reality: 10764,
        "Sci-Fi & Fantasy": 10765,
        Soap: 10766,
        Talk: 10767,
        "War & Politics": 10768,
        Western: 37,
      }
      const genreId = genreMap[genre]
      if (genreId) {
        url += `&with_genres=${genreId}`
      }
    }

    if (year && year !== "All Years") {
      url += `&first_air_date_year=${year}`
    }
  }

  try {
    const res = await fetch(url, { next: { revalidate: 3600 } })

    if (!res.ok) {
      console.error(`TMDB TV request failed (${res.status}) for ${url}`)
      return { results: [] }
    }

    const data = await res.json()

    // Filter adult content on server side as additional safety
    if (!showAdultContent && data.results) {
      data.results = data.results.filter((show: TVShow) => !show.adult)
    }

    return data
  } catch (err) {
    console.error("TMDB TV fetch error:", err)
    return { results: [] }
  }
}

export async function TVGrid(props: TVGridProps) {
  const data = await getTVShows(props)
  const shows: TVShow[] = data.results || []

  if (shows.length === 0) {
    return (
      <div className="text-center py-12">
        <p className="text-muted-foreground">No TV shows found.</p>
      </div>
    )
  }

  return (
    <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-6">
      {shows.map((show) => (
        <Link key={show.id} href={`/tv/${show.id}`}>
          <Card className="group hover:shadow-lg transition-shadow duration-200">
            <CardContent className="p-0">
              <div className="relative aspect-[2/3] overflow-hidden rounded-t-lg">
                {show.poster_path ? (
                  <Image
                    src={`https://image.tmdb.org/t/p/w500${show.poster_path}`}
                    alt={show.name}
                    fill
                    className="object-cover group-hover:scale-105 transition-transform duration-200"
                  />
                ) : (
                  <div className="w-full h-full bg-muted flex items-center justify-center">
                    <span className="text-muted-foreground">No Image</span>
                  </div>
                )}

                <div className="absolute top-2 right-2 flex flex-col gap-1">
                  <Badge variant="secondary" className="text-xs">
                    <Star className="w-3 h-3 mr-1 fill-yellow-400 text-yellow-400" />
                    {show.vote_average.toFixed(1)}
                  </Badge>

                  {show.adult && (
                    <Badge variant="destructive" className="text-xs">
                      <Shield className="w-3 h-3 mr-1" />
                      18+
                    </Badge>
                  )}
                </div>
              </div>

              <div className="p-3 space-y-2">
                <h3 className="font-semibold text-sm line-clamp-2 group-hover:text-primary">{show.name}</h3>

                {show.first_air_date && (
                  <div className="flex items-center text-xs text-muted-foreground">
                    <Calendar className="w-3 h-3 mr-1" />
                    {new Date(show.first_air_date).getFullYear()}
                  </div>
                )}
              </div>
            </CardContent>
          </Card>
        </Link>
      ))}
    </div>
  )
}
