const express = require("express")
const supabase = require("../lib/supabase")
const router = express.Router()

// Get all blog posts
router.get("/posts", async (req, res) => {
  try {
    const { page = 1, limit = 10 } = req.query
    const offset = (page - 1) * limit

    const { data, error } = await supabase
      .from("blog_posts")
      .select(`
        *,
        blog_categories(name, slug)
      `)
      .eq("published", true)
      .order("created_at", { ascending: false })
      .range(offset, offset + limit - 1)

    if (error) {
      return res.status(500).json({ error: error.message })
    }

    res.json(data)
  } catch (error) {
    console.error("Blog posts error:", error)
    res.status(500).json({ error: "Failed to get blog posts" })
  }
})

// Get single blog post
router.get("/posts/:slug", async (req, res) => {
  try {
    const { slug } = req.params

    const { data, error } = await supabase
      .from("blog_posts")
      .select(`
        *,
        blog_categories(name, slug)
      `)
      .eq("slug", slug)
      .eq("published", true)
      .single()

    if (error) {
      return res.status(404).json({ error: "Blog post not found" })
    }

    res.json(data)
  } catch (error) {
    console.error("Blog post error:", error)
    res.status(500).json({ error: "Failed to get blog post" })
  }
})

// Create blog post (admin only)
router.post("/posts", async (req, res) => {
  try {
    const { title, content, excerpt, category_id, published = false } = req.body

    // Generate slug from title
    const slug = title
      .toLowerCase()
      .replace(/[^a-z0-9 -]/g, "")
      .replace(/\s+/g, "-")
      .replace(/-+/g, "-")

    const { data, error } = await supabase
      .from("blog_posts")
      .insert({
        title,
        slug,
        content,
        excerpt,
        category_id,
        published,
      })
      .select()
      .single()

    if (error) {
      return res.status(400).json({ error: error.message })
    }

    res.status(201).json(data)
  } catch (error) {
    console.error("Create blog post error:", error)
    res.status(500).json({ error: "Failed to create blog post" })
  }
})

module.exports = router
