import { type NextRequest, NextResponse } from "next/server"
import { createRouteHandlerClient } from "@supabase/auth-helpers-nextjs"
import { cookies } from "next/headers"
import { generateText } from "ai"
import { openai } from "@ai-sdk/openai"

export async function POST(request: NextRequest) {
  try {
    const supabase = createRouteHandlerClient({ cookies })

    // Check if user is admin
    const {
      data: { session },
    } = await supabase.auth.getSession()

    if (!session) {
      return NextResponse.json({ error: "Unauthorized" }, { status: 401 })
    }

    const { data: profile } = await supabase.from("profiles").select("role").eq("id", session.user.id).single()

    if (profile?.role !== "admin") {
      return NextResponse.json({ error: "Forbidden" }, { status: 403 })
    }

    const { movieId, movieTitle, movieOverview, moviePoster, type = "trending" } = await request.json()

    if (!movieId || !movieTitle) {
      return NextResponse.json({ error: "Movie ID and title are required" }, { status: 400 })
    }

    // Generate blog post content using ChatGPT
    const prompt = `Write an engaging blog post about the ${type} movie "${movieTitle}". 

Movie Overview: ${movieOverview || "No overview available"}

Please write a comprehensive blog post that includes:
1. An attention-grabbing introduction
2. Why this movie is trending/popular
3. Key highlights about the plot (without major spoilers)
4. What makes it worth watching
5. Target audience
6. A compelling conclusion

The tone should be enthusiastic but informative, suitable for movie enthusiasts. 
The post should be around 500-800 words.
Use HTML formatting with proper paragraphs (<p>), headings (<h2>, <h3>), and emphasis (<strong>, <em>) where appropriate.`

    const { text: content } = await generateText({
      model: openai("gpt-4o"),
      prompt,
      maxTokens: 1500,
    })

    // Generate a shorter excerpt
    const excerptPrompt = `Based on this blog post content, write a compelling 2-3 sentence excerpt that would make people want to read the full article:

${content}`

    const { text: excerpt } = await generateText({
      model: openai("gpt-4o"),
      prompt: excerptPrompt,
      maxTokens: 200,
    })

    // Generate SEO-friendly slug
    const slug = movieTitle
      .toLowerCase()
      .replace(/[^a-z0-9\s-]/g, "")
      .replace(/\s+/g, "-")
      .replace(/-+/g, "-")
      .trim()

    // Create unique slug if needed
    const { data: existingPost } = await supabase.from("blog_posts").select("slug").eq("slug", slug).single()

    const finalSlug = existingPost ? `${slug}-${Date.now()}` : slug

    // Save blog post to database
    const { data: blogPost, error } = await supabase
      .from("blog_posts")
      .insert({
        title: `${movieTitle}: ${type === "trending" ? "Why This Movie is Trending Right Now" : "A Must-Watch Movie"}`,
        slug: finalSlug,
        content,
        excerpt,
        featured_image: moviePoster,
        movie_id: movieId,
        movie_title: movieTitle,
        movie_poster: moviePoster,
        status: "published",
        author_id: session.user.id,
        published_at: new Date().toISOString(),
      })
      .select()
      .single()

    if (error) {
      console.error("Error saving blog post:", error)
      return NextResponse.json({ error: "Failed to save blog post" }, { status: 500 })
    }

    // Add to trending category
    const { data: category } = await supabase
      .from("blog_categories")
      .select("id")
      .eq("slug", type === "trending" ? "trending" : "recommendations")
      .single()

    if (category) {
      await supabase.from("blog_post_categories").insert({
        blog_post_id: blogPost.id,
        category_id: category.id,
      })
    }

    return NextResponse.json({
      success: true,
      blogPost,
      message: "Blog post generated and published successfully!",
    })
  } catch (error) {
    console.error("Error generating blog post:", error)
    return NextResponse.json({ error: "Failed to generate blog post" }, { status: 500 })
  }
}
