import { Suspense } from "react"
import { Header } from "@/components/header"
import { MovieGrid } from "@/components/movie-grid"
import { TVGrid } from "@/components/tv-grid"
import { FilterBar } from "@/components/filter-bar"
import { ContentFilterNotice } from "@/components/content-filter-notice"
import { Button } from "@/components/ui/button"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { Film, Tv, TrendingUp, Star, Users, Calendar } from "lucide-react"
import Link from "next/link"

export default function HomePage({
  searchParams,
}: {
  searchParams: { [key: string]: string | string[] | undefined }
}) {
  const search = typeof searchParams.search === "string" ? searchParams.search : ""
  const type = typeof searchParams.type === "string" ? searchParams.type : "movie"
  const genre = typeof searchParams.genre === "string" ? searchParams.genre : ""
  const year = typeof searchParams.year === "string" ? searchParams.year : ""
  const sort = typeof searchParams.sort === "string" ? searchParams.sort : "popularity.desc"

  return (
    <div className="min-h-screen bg-background">
      <Header />

      {!search && (
        <section className="relative py-20 px-4 bg-gradient-to-br from-primary/10 via-background to-secondary/10">
          <div className="container mx-auto text-center">
            <div className="flex items-center justify-center mb-6">
              <Film className="h-12 w-12 text-primary mr-4" />
              <h1 className="text-4xl md:text-6xl font-bold bg-gradient-to-r from-primary to-secondary bg-clip-text text-transparent">
                MoviesWeb
              </h1>
            </div>
            <p className="text-xl md:text-2xl text-muted-foreground mb-8 max-w-3xl mx-auto">
              Your ultimate destination for discovering movies, TV shows, and entertainment content. Explore trending
              titles, get detailed information, and never miss what's worth watching.
            </p>
            <div className="flex flex-col sm:flex-row gap-4 justify-center">
              <Button size="lg" asChild>
                <Link href="/movies">
                  <Film className="mr-2 h-5 w-5" />
                  Browse Movies
                </Link>
              </Button>
              <Button size="lg" variant="outline" asChild>
                <Link href="/tv">
                  <Tv className="mr-2 h-5 w-5" />
                  Explore TV Shows
                </Link>
              </Button>
            </div>
          </div>
        </section>
      )}

      {!search && (
        <section className="py-16 px-4">
          <div className="container mx-auto">
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-12">
              <Card className="text-center">
                <CardHeader>
                  <TrendingUp className="h-8 w-8 text-primary mx-auto mb-2" />
                  <CardTitle>Trending Now</CardTitle>
                </CardHeader>
                <CardContent>
                  <CardDescription>Discover what's popular and trending in movies and TV shows</CardDescription>
                </CardContent>
              </Card>

              <Card className="text-center">
                <CardHeader>
                  <Star className="h-8 w-8 text-primary mx-auto mb-2" />
                  <CardTitle>Top Rated</CardTitle>
                </CardHeader>
                <CardContent>
                  <CardDescription>Find the highest-rated content according to critics and audiences</CardDescription>
                </CardContent>
              </Card>

              <Card className="text-center">
                <CardHeader>
                  <Users className="h-8 w-8 text-primary mx-auto mb-2" />
                  <CardTitle>Cast & Crew</CardTitle>
                </CardHeader>
                <CardContent>
                  <CardDescription>Explore detailed information about actors, directors, and creators</CardDescription>
                </CardContent>
              </Card>

              <Card className="text-center">
                <CardHeader>
                  <Calendar className="h-8 w-8 text-primary mx-auto mb-2" />
                  <CardTitle>Release Dates</CardTitle>
                </CardHeader>
                <CardContent>
                  <CardDescription>Stay updated with upcoming releases and premiere dates</CardDescription>
                </CardContent>
              </Card>
            </div>
          </div>
        </section>
      )}

      <div className="container mx-auto px-4 py-8">
        <ContentFilterNotice />

        {search && (
          <div className="mb-6">
            <h1 className="text-2xl font-bold mb-2">Search Results</h1>
            <p className="text-muted-foreground">Showing results for "{search}"</p>
          </div>
        )}

        <FilterBar />

        <Suspense fallback={<div className="text-center py-8">Loading...</div>}>
          {type === "tv" ? (
            <TVGrid searchQuery={search} genre={genre} year={year} sortBy={sort} />
          ) : (
            <MovieGrid searchQuery={search} genre={genre} year={year} sortBy={sort} />
          )}
        </Suspense>
      </div>
    </div>
  )
}
