"use client"

import { useSearchParams, useRouter } from "next/navigation"
import { Button } from "@/components/ui/button"
import { Badge } from "@/components/ui/badge"
import { X, Filter } from "lucide-react"

interface FilterBarProps {
  type: "movie" | "tv"
}

export function FilterBar({ type }: FilterBarProps) {
  const searchParams = useSearchParams()
  const router = useRouter()

  const activeFilters = {
    genre: searchParams.get("genre"),
    year: searchParams.get("year"),
    sort: searchParams.get("sort"),
    search: searchParams.get("search"),
  }

  const hasActiveFilters = Object.values(activeFilters).some(Boolean)

  const removeFilter = (filterKey: string) => {
    const params = new URLSearchParams(searchParams.toString())
    params.delete(filterKey)
    params.delete("page") // Reset to page 1

    const basePath = type === "movie" ? "/movies" : "/tv"
    router.push(`${basePath}?${params.toString()}`)
  }

  const clearAllFilters = () => {
    const basePath = type === "movie" ? "/movies" : "/tv"
    router.push(basePath)
  }

  const getSortLabel = (sortValue: string) => {
    const sortLabels: { [key: string]: string } = {
      "popularity.desc": "Most Popular",
      "popularity.asc": "Least Popular",
      "release_date.desc": "Newest First",
      "release_date.asc": "Oldest First",
      "first_air_date.desc": "Newest First",
      "first_air_date.asc": "Oldest First",
      "vote_average.desc": "Highest Rated",
      "vote_average.asc": "Lowest Rated",
      "title.asc": "A-Z",
      "title.desc": "Z-A",
      "name.asc": "A-Z",
      "name.desc": "Z-A",
    }
    return sortLabels[sortValue] || sortValue
  }

  if (!hasActiveFilters) {
    return null
  }

  return (
    <div className="flex flex-wrap items-center gap-2 p-4 bg-muted/30 rounded-lg border">
      <div className="flex items-center gap-2 mr-2">
        <Filter className="h-4 w-4 text-muted-foreground" />
        <span className="text-sm font-medium">Active Filters:</span>
      </div>

      {activeFilters.search && (
        <Badge variant="secondary" className="gap-1">
          Search: "{activeFilters.search}"
          <X className="h-3 w-3 cursor-pointer hover:text-destructive" onClick={() => removeFilter("search")} />
        </Badge>
      )}

      {activeFilters.genre && (
        <Badge variant="secondary" className="gap-1">
          Genre: {activeFilters.genre}
          <X className="h-3 w-3 cursor-pointer hover:text-destructive" onClick={() => removeFilter("genre")} />
        </Badge>
      )}

      {activeFilters.year && (
        <Badge variant="secondary" className="gap-1">
          Year: {activeFilters.year}
          <X className="h-3 w-3 cursor-pointer hover:text-destructive" onClick={() => removeFilter("year")} />
        </Badge>
      )}

      {activeFilters.sort && activeFilters.sort !== "popularity.desc" && (
        <Badge variant="secondary" className="gap-1">
          Sort: {getSortLabel(activeFilters.sort)}
          <X className="h-3 w-3 cursor-pointer hover:text-destructive" onClick={() => removeFilter("sort")} />
        </Badge>
      )}

      <Button
        variant="ghost"
        size="sm"
        onClick={clearAllFilters}
        className="ml-auto text-muted-foreground hover:text-foreground"
      >
        Clear All
      </Button>
    </div>
  )
}
