"use client"

import { useState } from "react"
import { useRouter, useSearchParams } from "next/navigation"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { Badge } from "@/components/ui/badge"
import { X } from "lucide-react"

const genres = [
  { id: 28, name: "Action" },
  { id: 12, name: "Adventure" },
  { id: 16, name: "Animation" },
  { id: 35, name: "Comedy" },
  { id: 80, name: "Crime" },
  { id: 99, name: "Documentary" },
  { id: 18, name: "Drama" },
  { id: 10751, name: "Family" },
  { id: 14, name: "Fantasy" },
  { id: 36, name: "History" },
  { id: 27, name: "Horror" },
  { id: 10402, name: "Music" },
  { id: 9648, name: "Mystery" },
  { id: 10749, name: "Romance" },
  { id: 878, name: "Science Fiction" },
  { id: 53, name: "Thriller" },
  { id: 10752, name: "War" },
  { id: 37, name: "Western" },
]

const years = Array.from({ length: 30 }, (_, i) => new Date().getFullYear() - i)

const sortOptions = [
  { value: "popularity.desc", label: "Most Popular" },
  { value: "popularity.asc", label: "Least Popular" },
  { value: "release_date.desc", label: "Newest First" },
  { value: "release_date.asc", label: "Oldest First" },
  { value: "vote_average.desc", label: "Highest Rated" },
  { value: "vote_average.asc", label: "Lowest Rated" },
  { value: "title.asc", label: "A-Z" },
  { value: "title.desc", label: "Z-A" },
]

export function MovieFilters() {
  const router = useRouter()
  const searchParams = useSearchParams()
  const [selectedGenre, setSelectedGenre] = useState(searchParams.get("genre") || "All Genres")
  const [selectedYear, setSelectedYear] = useState(searchParams.get("year") || "All Years")
  const [selectedSort, setSelectedSort] = useState(searchParams.get("sort") || "popularity.desc")

  const updateFilters = (key: string, value: string) => {
    const params = new URLSearchParams(searchParams.toString())

    if (value && value !== "All Genres" && value !== "All Years") {
      params.set(key, value)
    } else {
      params.delete(key)
    }

    // Reset to page 1 when filters change
    params.delete("page")

    router.push(`/movies?${params.toString()}`)
  }

  const clearFilters = () => {
    const params = new URLSearchParams(searchParams.toString())
    params.delete("genre")
    params.delete("year")
    params.delete("sort")
    params.delete("page")

    setSelectedGenre("All Genres")
    setSelectedYear("All Years")
    setSelectedSort("popularity.desc")

    router.push(`/movies?${params.toString()}`)
  }

  const hasActiveFilters =
    selectedGenre !== "All Genres" || selectedYear !== "All Years" || selectedSort !== "popularity.desc"

  return (
    <Card>
      <CardHeader>
        <div className="flex items-center justify-between">
          <CardTitle>Filters</CardTitle>
          {hasActiveFilters && (
            <Button variant="ghost" size="sm" onClick={clearFilters}>
              <X className="h-4 w-4 mr-1" />
              Clear
            </Button>
          )}
        </div>
      </CardHeader>
      <CardContent className="space-y-6">
        {/* Genre Filter */}
        <div>
          <label className="text-sm font-medium mb-2 block">Genre</label>
          <Select
            value={selectedGenre}
            onValueChange={(value) => {
              setSelectedGenre(value)
              updateFilters("genre", value === "All Genres" ? "" : value)
            }}
          >
            <SelectTrigger>
              <SelectValue placeholder="All Genres" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="All Genres">All Genres</SelectItem>
              {genres.map((genre) => (
                <SelectItem key={genre.id} value={genre.name}>
                  {genre.name}
                </SelectItem>
              ))}
            </SelectContent>
          </Select>
        </div>

        {/* Year Filter */}
        <div>
          <label className="text-sm font-medium mb-2 block">Release Year</label>
          <Select
            value={selectedYear}
            onValueChange={(value) => {
              setSelectedYear(value)
              updateFilters("year", value)
            }}
          >
            <SelectTrigger>
              <SelectValue placeholder="All Years" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="All Years">All Years</SelectItem>
              {years.map((year) => (
                <SelectItem key={year} value={year.toString()}>
                  {year}
                </SelectItem>
              ))}
            </SelectContent>
          </Select>
        </div>

        {/* Sort Filter */}
        <div>
          <label className="text-sm font-medium mb-2 block">Sort By</label>
          <Select
            value={selectedSort}
            onValueChange={(value) => {
              setSelectedSort(value)
              updateFilters("sort", value)
            }}
          >
            <SelectTrigger>
              <SelectValue />
            </SelectTrigger>
            <SelectContent>
              {sortOptions.map((option) => (
                <SelectItem key={option.value} value={option.value}>
                  {option.label}
                </SelectItem>
              ))}
            </SelectContent>
          </Select>
        </div>

        {/* Active Filters */}
        {hasActiveFilters && (
          <div>
            <label className="text-sm font-medium mb-2 block">Active Filters</label>
            <div className="flex flex-wrap gap-2">
              {selectedGenre !== "All Genres" && (
                <Badge variant="secondary" className="text-xs">
                  {selectedGenre}
                  <X
                    className="h-3 w-3 ml-1 cursor-pointer"
                    onClick={() => {
                      setSelectedGenre("All Genres")
                      updateFilters("genre", "")
                    }}
                  />
                </Badge>
              )}
              {selectedYear !== "All Years" && (
                <Badge variant="secondary" className="text-xs">
                  {selectedYear}
                  <X
                    className="h-3 w-3 ml-1 cursor-pointer"
                    onClick={() => {
                      setSelectedYear("All Years")
                      updateFilters("year", "")
                    }}
                  />
                </Badge>
              )}
              {selectedSort !== "popularity.desc" && (
                <Badge variant="secondary" className="text-xs">
                  {sortOptions.find((opt) => opt.value === selectedSort)?.label}
                  <X
                    className="h-3 w-3 ml-1 cursor-pointer"
                    onClick={() => {
                      setSelectedSort("popularity.desc")
                      updateFilters("sort", "popularity.desc")
                    }}
                  />
                </Badge>
              )}
            </div>
          </div>
        )}
      </CardContent>
    </Card>
  )
}
