import Image from "next/image"
import Link from "next/link"
import { Card, CardContent } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import { Star } from "lucide-react"

interface Movie {
  id: number
  title: string
  poster_path: string | null
  vote_average: number
}

async function getSimilarMovies(movieId: string) {
  try {
    const res = await fetch(
      `https://api.themoviedb.org/3/movie/${movieId}/similar?api_key=${process.env.TMDB_API_KEY}`,
      { next: { revalidate: 3600 } },
    )

    if (!res.ok) {
      console.error(`TMDB similar-movies fetch failed (${res.status}) for id ${movieId}`)
      return { results: [] }
    }

    return res.json()
  } catch (err) {
    console.error("TMDB similar-movies fetch error:", err)
    return { results: [] }
  }
}

export async function SimilarMovies({ movieId }: { movieId: string }) {
  const data = await getSimilarMovies(movieId)
  const movies: Movie[] = data.results?.slice(0, 6) || []

  if (movies.length === 0) {
    return <p className="text-muted-foreground">No similar movies found.</p>
  }

  return (
    <div className="grid grid-cols-2 md:grid-cols-3 gap-4">
      {movies.map((movie) => (
        <Link key={movie.id} href={`/movie/${movie.id}`}>
          <Card className="group hover:shadow-lg transition-shadow duration-200">
            <CardContent className="p-0">
              <div className="relative aspect-[2/3] overflow-hidden rounded-t-lg">
                {movie.poster_path ? (
                  <Image
                    src={`https://image.tmdb.org/t/p/w300${movie.poster_path}`}
                    alt={movie.title}
                    fill
                    className="object-cover group-hover:scale-105 transition-transform duration-200"
                  />
                ) : (
                  <div className="w-full h-full bg-muted flex items-center justify-center">
                    <span className="text-muted-foreground text-xs">No Image</span>
                  </div>
                )}

                <div className="absolute top-2 right-2">
                  <Badge variant="secondary" className="text-xs">
                    <Star className="w-3 h-3 mr-1 fill-yellow-400 text-yellow-400" />
                    {movie.vote_average.toFixed(1)}
                  </Badge>
                </div>
              </div>

              <div className="p-3">
                <h3 className="font-semibold text-sm line-clamp-2 group-hover:text-primary">{movie.title}</h3>
              </div>
            </CardContent>
          </Card>
        </Link>
      ))}
    </div>
  )
}
