"use client"

import { useState } from "react"
import { useRouter, useSearchParams } from "next/navigation"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { Badge } from "@/components/ui/badge"
import { X } from "lucide-react"

const tvGenres = [
  { id: 10759, name: "Action & Adventure" },
  { id: 16, name: "Animation" },
  { id: 35, name: "Comedy" },
  { id: 80, name: "Crime" },
  { id: 99, name: "Documentary" },
  { id: 18, name: "Drama" },
  { id: 10751, name: "Family" },
  { id: 10762, name: "Kids" },
  { id: 9648, name: "Mystery" },
  { id: 10763, name: "News" },
  { id: 10764, name: "Reality" },
  { id: 10765, name: "Sci-Fi & Fantasy" },
  { id: 10766, name: "Soap" },
  { id: 10767, name: "Talk" },
  { id: 10768, name: "War & Politics" },
  { id: 37, name: "Western" },
]

const years = Array.from({ length: 30 }, (_, i) => new Date().getFullYear() - i)

const sortOptions = [
  { value: "popularity.desc", label: "Most Popular" },
  { value: "popularity.asc", label: "Least Popular" },
  { value: "first_air_date.desc", label: "Newest First" },
  { value: "first_air_date.asc", label: "Oldest First" },
  { value: "vote_average.desc", label: "Highest Rated" },
  { value: "vote_average.asc", label: "Lowest Rated" },
  { value: "name.asc", label: "A-Z" },
  { value: "name.desc", label: "Z-A" },
]

export function TVFilters() {
  const router = useRouter()
  const searchParams = useSearchParams()
  const [selectedGenre, setSelectedGenre] = useState(searchParams.get("genre") || "All Genres")
  const [selectedYear, setSelectedYear] = useState(searchParams.get("year") || "All Years")
  const [selectedSort, setSelectedSort] = useState(searchParams.get("sort") || "popularity.desc")

  const updateFilters = (key: string, value: string) => {
    const params = new URLSearchParams(searchParams.toString())

    if (value && value !== "All Genres" && value !== "All Years") {
      params.set(key, value)
    } else {
      params.delete(key)
    }

    // Reset to page 1 when filters change
    params.delete("page")

    router.push(`/tv?${params.toString()}`)
  }

  const clearFilters = () => {
    const params = new URLSearchParams(searchParams.toString())
    params.delete("genre")
    params.delete("year")
    params.delete("sort")
    params.delete("page")

    setSelectedGenre("All Genres")
    setSelectedYear("All Years")
    setSelectedSort("popularity.desc")

    router.push(`/tv?${params.toString()}`)
  }

  const hasActiveFilters =
    selectedGenre !== "All Genres" || selectedYear !== "All Years" || selectedSort !== "popularity.desc"

  return (
    <Card>
      <CardHeader>
        <div className="flex items-center justify-between">
          <CardTitle>Filters</CardTitle>
          {hasActiveFilters && (
            <Button variant="ghost" size="sm" onClick={clearFilters}>
              <X className="h-4 w-4 mr-1" />
              Clear
            </Button>
          )}
        </div>
      </CardHeader>
      <CardContent className="space-y-6">
        {/* Genre Filter */}
        <div>
          <label className="text-sm font-medium mb-2 block">Genre</label>
          <Select
            value={selectedGenre}
            onValueChange={(value) => {
              setSelectedGenre(value)
              updateFilters("genre", value === "All Genres" ? "" : value)
            }}
          >
            <SelectTrigger>
              <SelectValue placeholder="All Genres" />
            </SelectTrigger>
            <SelectContent>
              {tvGenres.map((genre) => (
                <SelectItem key={genre.id} value={genre.name}>
                  {genre.name}
                </SelectItem>
              ))}
            </SelectContent>
          </Select>
        </div>

        {/* Year Filter */}
        <div>
          <label className="text-sm font-medium mb-2 block">First Air Date</label>
          <Select
            value={selectedYear}
            onValueChange={(value) => {
              setSelectedYear(value)
              updateFilters("year", value === "All Years" ? "" : value)
            }}
          >
            <SelectTrigger>
              <SelectValue placeholder="All Years" />
            </SelectTrigger>
            <SelectContent>
              {years.map((year) => (
                <SelectItem key={year} value={year.toString()}>
                  {year}
                </SelectItem>
              ))}
            </SelectContent>
          </Select>
        </div>

        {/* Sort Filter */}
        <div>
          <label className="text-sm font-medium mb-2 block">Sort By</label>
          <Select
            value={selectedSort}
            onValueChange={(value) => {
              setSelectedSort(value)
              updateFilters("sort", value)
            }}
          >
            <SelectTrigger>
              <SelectValue />
            </SelectTrigger>
            <SelectContent>
              {sortOptions.map((option) => (
                <SelectItem key={option.value} value={option.value}>
                  {option.label}
                </SelectItem>
              ))}
            </SelectContent>
          </Select>
        </div>

        {/* Active Filters */}
        {hasActiveFilters && (
          <div>
            <label className="text-sm font-medium mb-2 block">Active Filters</label>
            <div className="flex flex-wrap gap-2">
              {selectedGenre !== "All Genres" && (
                <Badge variant="secondary" className="text-xs">
                  {selectedGenre}
                  <X
                    className="h-3 w-3 ml-1 cursor-pointer"
                    onClick={() => {
                      setSelectedGenre("All Genres")
                      updateFilters("genre", "")
                    }}
                  />
                </Badge>
              )}
              {selectedYear !== "All Years" && (
                <Badge variant="secondary" className="text-xs">
                  {selectedYear}
                  <X
                    className="h-3 w-3 ml-1 cursor-pointer"
                    onClick={() => {
                      setSelectedYear("All Years")
                      updateFilters("year", "")
                    }}
                  />
                </Badge>
              )}
              {selectedSort !== "popularity.desc" && (
                <Badge variant="secondary" className="text-xs">
                  {sortOptions.find((opt) => opt.value === selectedSort)?.label}
                  <X
                    className="h-3 w-3 ml-1 cursor-pointer"
                    onClick={() => {
                      setSelectedSort("popularity.desc")
                      updateFilters("sort", "popularity.desc")
                    }}
                  />
                </Badge>
              )}
            </div>
          </div>
        )}
      </CardContent>
    </Card>
  )
}
