import { Suspense } from "react"
import { Header } from "@/components/header"
import { TVGrid } from "@/components/tv-grid"
import { TVFilters } from "@/components/tv-filters"
import { QuickFilters } from "@/components/quick-filters"
import { MobileFilterToggle } from "@/components/mobile-filter-toggle"
import { FilterBar } from "@/components/filter-bar"
import { ContentFilterNotice } from "@/components/content-filter-notice"
import { Skeleton } from "@/components/ui/skeleton"
import { SlidersHorizontal } from "lucide-react"
import { createServerClient } from "@/lib/supabase"

export default async function TVPage({
  searchParams,
}: {
  searchParams: {
    search?: string
    genre?: string
    year?: string
    sort?: string
    page?: string
  }
}) {
  const search = searchParams.search || ""
  const genre = searchParams.genre || ""
  const year = searchParams.year || ""
  const sort = searchParams.sort || "popularity.desc"
  const page = Number.parseInt(searchParams.page || "1")

  const supabase = createServerClient()
  const {
    data: { session },
  } = await supabase.auth.getSession()
  const showAdultContent = !!session?.user

  return (
    <div className="min-h-screen bg-background">
      <Header />

      {/* Hero Section */}
      <section className="bg-gradient-to-r from-green-900 via-teal-900 to-blue-900 py-16">
        <div className="container mx-auto px-4">
          <div className="flex flex-col md:flex-row md:items-center md:justify-between">
            <div>
              <h1 className="text-4xl font-bold text-white mb-4">TV Shows & Series</h1>
              <p className="text-xl text-gray-200 max-w-2xl">
                Binge-watch the best TV shows and series from around the world
              </p>
            </div>
            <div className="mt-4 md:mt-0">
              <MobileFilterToggle type="tv" />
            </div>
          </div>
        </div>
      </section>

      <div className="container mx-auto px-4 py-8">
        {/* Content Filter Notice */}
        <ContentFilterNotice />

        {/* Filter Bar */}
        <FilterBar type="tv" />

        <div className="grid grid-cols-1 lg:grid-cols-4 gap-8 mt-6">
          {/* Desktop Filters Sidebar */}
          <div className="lg:col-span-1 hidden lg:block">
            <div className="sticky top-24 space-y-6">
              <QuickFilters type="tv" />
              <div className="border-t pt-6">
                <div className="flex items-center gap-2 mb-4">
                  <SlidersHorizontal className="h-5 w-5" />
                  <h3 className="text-lg font-semibold">Advanced Filters</h3>
                </div>
                <TVFilters />
              </div>
            </div>
          </div>

          {/* TV Shows Grid */}
          <div className="lg:col-span-3">
            <div className="mb-6">
              <h2 className="text-2xl font-bold mb-2">{search ? `Search Results for "${search}"` : "All TV Shows"}</h2>
              <p className="text-muted-foreground">
                {genre && `Genre: ${genre} • `}
                {year && `Year: ${year} • `}
                Sorted by {sort.replace(".", " ").replace("_", " ")}
              </p>
            </div>

            <Suspense fallback={<TVGridSkeleton />}>
              <TVGrid
                search={search}
                genre={genre}
                year={year}
                sort={sort}
                page={page}
                showAdultContent={showAdultContent}
              />
            </Suspense>
          </div>
        </div>
      </div>
    </div>
  )
}

function TVGridSkeleton() {
  return (
    <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-6">
      {Array.from({ length: 12 }).map((_, i) => (
        <div key={i} className="space-y-3">
          <Skeleton className="aspect-[2/3] w-full rounded-lg" />
          <Skeleton className="h-4 w-full" />
          <Skeleton className="h-3 w-2/3" />
        </div>
      ))}
    </div>
  )
}

export const metadata = {
  title: "TV Shows - MoviesWeb",
  description: "Discover and explore TV shows and series from every genre",
}
