"use client"

import { useState, useEffect } from "react"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { Badge } from "@/components/ui/badge"
import { useToast } from "@/hooks/use-toast"
import { Search, Plus, Edit, Trash2, RefreshCw } from "lucide-react"
import { createClient } from "@/lib/supabase"
import { BlogPostGenerator } from "@/components/blog-post-generator"

export function AdminDashboard() {
  const [searchQuery, setSearchQuery] = useState("")
  const [searchResults, setSearchResults] = useState([])
  const [loading, setLoading] = useState(false)
  const [savedMovies, setSavedMovies] = useState([])
  const [trendingMovies, setTrendingMovies] = useState([])
  const supabase = createClient()
  const { toast } = useToast()

  useEffect(() => {
    fetchSavedMovies()
    fetchTrendingMovies()
  }, [])

  const fetchTrendingMovies = async () => {
    try {
      const response = await fetch(
        `https://api.themoviedb.org/3/trending/movie/week?api_key=${process.env.NEXT_PUBLIC_TMDB_API_KEY}`,
      )
      const data = await response.json()
      setTrendingMovies(data.results?.slice(0, 10) || [])
    } catch (error) {
      console.error("Error fetching trending movies:", error)
    }
  }

  const fetchSavedMovies = async () => {
    try {
      const { data, error } = await supabase.from("movies").select("*").order("created_at", { ascending: false })

      if (error) throw error
      setSavedMovies(data || [])
    } catch (error) {
      console.error("Error fetching saved movies:", error)
    }
  }

  const searchTMDB = async () => {
    if (!searchQuery.trim()) return

    setLoading(true)
    try {
      const response = await fetch(`/api/tmdb/search?query=${encodeURIComponent(searchQuery)}`)
      const data = await response.json()

      if (data.error) {
        throw new Error(data.error)
      }

      setSearchResults(data.results || [])
    } catch (error) {
      toast({
        title: "Error",
        description: "Failed to search movies",
        variant: "destructive",
      })
    } finally {
      setLoading(false)
    }
  }

  const saveMovie = async (movie: any) => {
    try {
      const { error } = await supabase.from("movies").upsert({
        tmdb_id: movie.id,
        title: movie.title,
        overview: movie.overview,
        poster_path: movie.poster_path,
        backdrop_path: movie.backdrop_path,
        release_date: movie.release_date,
        vote_average: movie.vote_average,
        vote_count: movie.vote_count,
        popularity: movie.popularity,
        adult: movie.adult,
        original_language: movie.original_language,
        original_title: movie.original_title,
        genre_ids: movie.genre_ids,
      })

      if (error) throw error

      toast({
        title: "Success",
        description: "Movie saved successfully",
      })

      fetchSavedMovies()
    } catch (error) {
      toast({
        title: "Error",
        description: "Failed to save movie",
        variant: "destructive",
      })
    }
  }

  const deleteMovie = async (id: number) => {
    try {
      const { error } = await supabase.from("movies").delete().eq("id", id)

      if (error) throw error

      toast({
        title: "Success",
        description: "Movie deleted successfully",
      })

      fetchSavedMovies()
    } catch (error) {
      toast({
        title: "Error",
        description: "Failed to delete movie",
        variant: "destructive",
      })
    }
  }

  const syncWithTMDB = async () => {
    setLoading(true)
    try {
      const response = await fetch("/api/tmdb/sync", { method: "POST" })
      const data = await response.json()

      if (data.error) {
        throw new Error(data.error)
      }

      toast({
        title: "Success",
        description: `Synced ${data.count} movies from TMDB`,
      })

      fetchSavedMovies()
    } catch (error) {
      toast({
        title: "Error",
        description: "Failed to sync with TMDB",
        variant: "destructive",
      })
    } finally {
      setLoading(false)
    }
  }

  return (
    <div className="space-y-8">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold">Admin Dashboard</h1>
          <p className="text-muted-foreground">Manage movie database content</p>
        </div>
        <Button onClick={syncWithTMDB} disabled={loading}>
          <RefreshCw className={`mr-2 h-4 w-4 ${loading ? "animate-spin" : ""}`} />
          Sync with TMDB
        </Button>
      </div>

      <Tabs defaultValue="search" className="space-y-6">
        <TabsList>
          <TabsTrigger value="search">Search & Add</TabsTrigger>
          <TabsTrigger value="manage">Manage Movies</TabsTrigger>
          <TabsTrigger value="blog">Blog Posts</TabsTrigger>
        </TabsList>

        <TabsContent value="search" className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle>Search TMDB</CardTitle>
              <CardDescription>Search for movies on TMDB and add them to your database</CardDescription>
            </CardHeader>
            <CardContent>
              <div className="flex space-x-2">
                <div className="flex-1">
                  <Label htmlFor="search">Movie Title</Label>
                  <Input
                    id="search"
                    value={searchQuery}
                    onChange={(e) => setSearchQuery(e.target.value)}
                    placeholder="Enter movie title..."
                    onKeyPress={(e) => e.key === "Enter" && searchTMDB()}
                  />
                </div>
                <div className="flex items-end">
                  <Button onClick={searchTMDB} disabled={loading}>
                    <Search className="mr-2 h-4 w-4" />
                    Search
                  </Button>
                </div>
              </div>
            </CardContent>
          </Card>

          {searchResults.length > 0 && (
            <Card>
              <CardHeader>
                <CardTitle>Search Results</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  {searchResults.map((movie: any) => (
                    <div key={movie.id} className="flex items-center space-x-4 p-4 border rounded-lg">
                      <div className="w-16 h-24 bg-muted rounded flex-shrink-0">
                        {movie.poster_path && (
                          <img
                            src={`https://image.tmdb.org/t/p/w92${movie.poster_path}`}
                            alt={movie.title}
                            className="w-full h-full object-cover rounded"
                          />
                        )}
                      </div>
                      <div className="flex-1 min-w-0">
                        <h3 className="font-semibold truncate">{movie.title}</h3>
                        <p className="text-sm text-muted-foreground">
                          {movie.release_date ? new Date(movie.release_date).getFullYear() : "N/A"}
                        </p>
                        <p className="text-sm text-muted-foreground line-clamp-2">{movie.overview}</p>
                        <div className="flex items-center space-x-2 mt-2">
                          <Badge variant="secondary">⭐ {movie.vote_average?.toFixed(1) || "N/A"}</Badge>
                        </div>
                      </div>
                      <Button onClick={() => saveMovie(movie)}>
                        <Plus className="mr-2 h-4 w-4" />
                        Add
                      </Button>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>
          )}
        </TabsContent>

        <TabsContent value="manage" className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle>Saved Movies ({savedMovies.length})</CardTitle>
              <CardDescription>Manage movies in your database</CardDescription>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                {savedMovies.map((movie: any) => (
                  <div key={movie.id} className="flex items-center space-x-4 p-4 border rounded-lg">
                    <div className="w-16 h-24 bg-muted rounded flex-shrink-0">
                      {movie.poster_path && (
                        <img
                          src={`https://image.tmdb.org/t/p/w92${movie.poster_path}`}
                          alt={movie.title}
                          className="w-full h-full object-cover rounded"
                        />
                      )}
                    </div>
                    <div className="flex-1 min-w-0">
                      <h3 className="font-semibold truncate">{movie.title}</h3>
                      <p className="text-sm text-muted-foreground">
                        {movie.release_date ? new Date(movie.release_date).getFullYear() : "N/A"}
                      </p>
                      <p className="text-sm text-muted-foreground line-clamp-2">{movie.overview}</p>
                      <div className="flex items-center space-x-2 mt-2">
                        <Badge variant="secondary">⭐ {movie.vote_average?.toFixed(1) || "N/A"}</Badge>
                        <Badge variant="outline">TMDB ID: {movie.tmdb_id}</Badge>
                      </div>
                    </div>
                    <div className="flex space-x-2">
                      <Button variant="outline" size="sm">
                        <Edit className="h-4 w-4" />
                      </Button>
                      <Button variant="destructive" size="sm" onClick={() => deleteMovie(movie.id)}>
                        <Trash2 className="h-4 w-4" />
                      </Button>
                    </div>
                  </div>
                ))}

                {savedMovies.length === 0 && (
                  <div className="text-center py-8">
                    <p className="text-muted-foreground">No movies saved yet.</p>
                  </div>
                )}
              </div>
            </CardContent>
          </Card>
        </TabsContent>
        <TabsContent value="blog" className="space-y-6">
          <BlogPostGenerator
            movies={trendingMovies}
            onPostGenerated={() => {
              toast({
                title: "Success",
                description: "Blog post generated successfully!",
              })
            }}
          />
        </TabsContent>
      </Tabs>
    </div>
  )
}
