import Image from "next/image"
import { Badge } from "@/components/ui/badge"
import { Card, CardContent } from "@/components/ui/card"
import { Calendar, MapPin, User, Award, ExternalLink } from "lucide-react"
import { Button } from "@/components/ui/button"

interface PersonDetailsProps {
  person: {
    id: number
    name: string
    biography: string
    profile_path: string | null
    birthday: string | null
    deathday: string | null
    place_of_birth: string | null
    known_for_department: string
    popularity: number
    also_known_as: string[]
    homepage: string | null
    imdb_id: string | null
    gender: number
  }
}

export function PersonDetails({ person }: PersonDetailsProps) {
  const age = person.birthday
    ? Math.floor((new Date().getTime() - new Date(person.birthday).getTime()) / (1000 * 60 * 60 * 24 * 365.25))
    : null

  const genderMap = {
    0: "Not specified",
    1: "Female",
    2: "Male",
    3: "Non-binary",
  }

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString("en-US", {
      year: "numeric",
      month: "long",
      day: "numeric",
    })
  }

  return (
    <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
      {/* Profile Image and Basic Info */}
      <div className="lg:col-span-1">
        <Card className="overflow-hidden">
          <CardContent className="p-0">
            <div className="relative aspect-[2/3]">
              {person.profile_path ? (
                <Image
                  src={`https://image.tmdb.org/t/p/w500${person.profile_path}`}
                  alt={person.name}
                  fill
                  className="object-cover"
                />
              ) : (
                <div className="w-full h-full bg-muted flex items-center justify-center">
                  <User className="h-24 w-24 text-muted-foreground" />
                </div>
              )}
            </div>
          </CardContent>
        </Card>

        {/* Personal Information Card */}
        <Card className="mt-6">
          <CardContent className="p-6 space-y-4">
            <h3 className="text-lg font-semibold">Personal Info</h3>

            <div className="space-y-3">
              <div>
                <h4 className="text-sm font-medium text-muted-foreground mb-1">Known For</h4>
                <Badge variant="secondary">{person.known_for_department}</Badge>
              </div>

              <div>
                <h4 className="text-sm font-medium text-muted-foreground mb-1">Gender</h4>
                <p className="text-sm">{genderMap[person.gender as keyof typeof genderMap]}</p>
              </div>

              {person.birthday && (
                <div>
                  <h4 className="text-sm font-medium text-muted-foreground mb-1">Birthday</h4>
                  <div className="flex items-center text-sm">
                    <Calendar className="w-4 h-4 mr-2" />
                    {formatDate(person.birthday)}
                    {age && !person.deathday && <span className="ml-2 text-muted-foreground">({age} years old)</span>}
                  </div>
                </div>
              )}

              {person.deathday && (
                <div>
                  <h4 className="text-sm font-medium text-muted-foreground mb-1">Died</h4>
                  <div className="flex items-center text-sm">
                    <Calendar className="w-4 h-4 mr-2" />
                    {formatDate(person.deathday)}
                    {person.birthday && (
                      <span className="ml-2 text-muted-foreground">
                        (aged{" "}
                        {Math.floor(
                          (new Date(person.deathday).getTime() - new Date(person.birthday).getTime()) /
                            (1000 * 60 * 60 * 24 * 365.25),
                        )}
                        )
                      </span>
                    )}
                  </div>
                </div>
              )}

              {person.place_of_birth && (
                <div>
                  <h4 className="text-sm font-medium text-muted-foreground mb-1">Place of Birth</h4>
                  <div className="flex items-center text-sm">
                    <MapPin className="w-4 h-4 mr-2" />
                    {person.place_of_birth}
                  </div>
                </div>
              )}

              {person.also_known_as && person.also_known_as.length > 0 && (
                <div>
                  <h4 className="text-sm font-medium text-muted-foreground mb-1">Also Known As</h4>
                  <div className="space-y-1">
                    {person.also_known_as.slice(0, 5).map((name, index) => (
                      <p key={index} className="text-sm">
                        {name}
                      </p>
                    ))}
                    {person.also_known_as.length > 5 && (
                      <p className="text-xs text-muted-foreground">+{person.also_known_as.length - 5} more</p>
                    )}
                  </div>
                </div>
              )}
            </div>

            {/* External Links */}
            <div className="pt-4 border-t space-y-2">
              {person.homepage && (
                <Button variant="outline" size="sm" asChild className="w-full bg-transparent">
                  <a href={person.homepage} target="_blank" rel="noopener noreferrer">
                    <ExternalLink className="w-4 h-4 mr-2" />
                    Official Website
                  </a>
                </Button>
              )}

              {person.imdb_id && (
                <Button variant="outline" size="sm" asChild className="w-full bg-transparent">
                  <a href={`https://www.imdb.com/name/${person.imdb_id}`} target="_blank" rel="noopener noreferrer">
                    <ExternalLink className="w-4 h-4 mr-2" />
                    IMDb Profile
                  </a>
                </Button>
              )}
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Biography and Details */}
      <div className="lg:col-span-2 space-y-6">
        <div>
          <h1 className="text-4xl font-bold mb-4">{person.name}</h1>

          <div className="flex items-center gap-4 mb-6">
            <Badge variant="secondary" className="text-sm">
              <Award className="w-4 h-4 mr-1" />
              Popularity: {person.popularity.toFixed(1)}
            </Badge>
          </div>

          {person.biography ? (
            <div className="space-y-4">
              <h2 className="text-2xl font-semibold">Biography</h2>
              <div className="prose prose-gray dark:prose-invert max-w-none">
                {person.biography.split("\n\n").map((paragraph, index) => (
                  <p key={index} className="text-muted-foreground leading-relaxed mb-4">
                    {paragraph}
                  </p>
                ))}
              </div>
            </div>
          ) : (
            <div className="text-center py-12">
              <User className="h-12 w-12 text-muted-foreground mx-auto mb-4" />
              <h3 className="text-lg font-semibold mb-2">No Biography Available</h3>
              <p className="text-muted-foreground">We don't have a biography for {person.name} yet.</p>
            </div>
          )}
        </div>
      </div>
    </div>
  )
}
