import Image from "next/image"
import Link from "next/link"
import { Card, CardContent } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import { Star } from "lucide-react"

interface TVShow {
  id: number
  name: string
  poster_path: string | null
  vote_average: number
}

async function getSimilarTVShows(tvShowId: string) {
  try {
    const res = await fetch(`https://api.themoviedb.org/3/tv/${tvShowId}/similar?api_key=${process.env.TMDB_API_KEY}`, {
      next: { revalidate: 3600 },
    })

    if (!res.ok) {
      console.error(`TMDB similar TV shows fetch failed (${res.status}) for id ${tvShowId}`)
      return { results: [] }
    }

    return res.json()
  } catch (err) {
    console.error("TMDB similar TV shows fetch error:", err)
    return { results: [] }
  }
}

export async function SimilarTVShows({ tvShowId }: { tvShowId: string }) {
  const data = await getSimilarTVShows(tvShowId)
  const shows: TVShow[] = data.results?.slice(0, 6) || []

  if (shows.length === 0) {
    return <p className="text-muted-foreground">No similar TV shows found.</p>
  }

  return (
    <div className="grid grid-cols-2 md:grid-cols-3 gap-4">
      {shows.map((show) => (
        <Link key={show.id} href={`/tv/${show.id}`}>
          <Card className="group hover:shadow-lg transition-shadow duration-200">
            <CardContent className="p-0">
              <div className="relative aspect-[2/3] overflow-hidden rounded-t-lg">
                {show.poster_path ? (
                  <Image
                    src={`https://image.tmdb.org/t/p/w300${show.poster_path}`}
                    alt={show.name}
                    fill
                    className="object-cover group-hover:scale-105 transition-transform duration-200"
                  />
                ) : (
                  <div className="w-full h-full bg-muted flex items-center justify-center">
                    <span className="text-muted-foreground text-xs">No Image</span>
                  </div>
                )}

                <div className="absolute top-2 right-2">
                  <Badge variant="secondary" className="text-xs">
                    <Star className="w-3 h-3 mr-1 fill-yellow-400 text-yellow-400" />
                    {show.vote_average.toFixed(1)}
                  </Badge>
                </div>
              </div>

              <div className="p-3">
                <h3 className="font-semibold text-sm line-clamp-2 group-hover:text-primary">{show.name}</h3>
              </div>
            </CardContent>
          </Card>
        </Link>
      ))}
    </div>
  )
}
