import Image from "next/image"
import { Card, CardContent } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import { Calendar, Star, Users } from "lucide-react"

interface TVSeasonDetailsProps {
  season: {
    id: number
    name: string
    overview: string
    poster_path: string | null
    season_number: number
    air_date: string
    vote_average: number
    episodes: any[]
  }
  tvShow: {
    name: string
  }
}

export function TVSeasonDetails({ season, tvShow }: TVSeasonDetailsProps) {
  return (
    <Card className="overflow-hidden">
      <CardContent className="p-0">
        <div className="flex flex-col md:flex-row">
          {/* Season Poster */}
          <div className="md:w-64 flex-shrink-0">
            <div className="relative aspect-[2/3]">
              {season.poster_path ? (
                <Image
                  src={`https://image.tmdb.org/t/p/w500${season.poster_path}`}
                  alt={season.name}
                  fill
                  className="object-cover"
                />
              ) : (
                <div className="w-full h-full bg-muted flex items-center justify-center">
                  <span className="text-muted-foreground">No Image</span>
                </div>
              )}
            </div>
          </div>

          {/* Season Info */}
          <div className="flex-1 p-6">
            <div className="mb-6">
              <h1 className="text-3xl font-bold mb-4">{season.name}</h1>

              <div className="flex flex-wrap items-center gap-3 mb-6">
                <Badge variant="secondary">Season {season.season_number}</Badge>

                <Badge variant="outline">
                  <Users className="w-3 h-3 mr-1" />
                  {season.episodes?.length || 0} Episodes
                </Badge>

                {season.air_date && (
                  <Badge variant="outline">
                    <Calendar className="w-3 h-3 mr-1" />
                    {new Date(season.air_date).toLocaleDateString("en-US", {
                      year: "numeric",
                      month: "long",
                      day: "numeric",
                    })}
                  </Badge>
                )}

                {season.vote_average > 0 && (
                  <Badge variant="secondary">
                    <Star className="w-3 h-3 mr-1 fill-yellow-400 text-yellow-400" />
                    {season.vote_average.toFixed(1)}
                  </Badge>
                )}
              </div>

              {season.overview ? (
                <p className="text-muted-foreground leading-relaxed">{season.overview}</p>
              ) : (
                <p className="text-muted-foreground italic">No description available for this season.</p>
              )}
            </div>
          </div>
        </div>
      </CardContent>
    </Card>
  )
}
